<?php

namespace App\Http\Controllers;

use App\DetalleRemision;
use App\Kardex;
use App\Libro;
use App\NotasRemision;
use App\Vendedor;
use Carbon\Carbon;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Session;
use Barryvdh\DomPDF\Facade as PDF;
use App\Exports\RelacionNotasRemisionExport;



class NotasRemisionController extends Controller
{
    /**
     * Display a listing of the resource.
     *
     * @return \Illuminate\Http\Response
     */
    public function index(Request $request)
    {
        $mes = date('m');
        $anio = date('Y');
        $imprimir = false;
        $remision = 0;



        if (isset($request->mes)) {
            $mes = $request->mes;
            $anio = $request->anio;
            $imprimir = $request->imprimir;
            $remision = $request->r;
        }
        $datos = NotasRemision::whereMonth('fecha', '=', $mes)
            ->whereYear('fecha', '=', $anio)
            ->orderBy('created_at', 'DESC')
            ->get();

        //        return dd($anio);
        return view('almacen.remision.index', [
            'datos' => $datos,
            'mes' => $mes,
            'anio' => $anio,
            'imprimir' => $imprimir,
            'remision' => $remision
        ]);
    }

    /**
     * Show the form for creating a new resource.
     *
     * @return \Illuminate\Http\Response
     */
    public function create()
    {
        $vendedores = Vendedor::orderBy('nombres', 'ASC')->get();
        $libros = Libro::orderBy('codigo', 'ASC')->get();
        return view('almacen.remision.create_edit', [
            'libros' => $libros,
            'vendedores' => $vendedores
        ]);
    }

    /**
     * Store a newly created resource in storage.
     *
     * @param  \Illuminate\Http\Request  $request
     * @return \Illuminate\Http\Response
     */
    public function store(Request $request)
    {
        //        return dd($request->all());

        try {
            DB::beginTransaction();
            if (isset($request->id)) {

                $r = NotasRemision::find($request->id);
                $r->detalle()->delete();
            } else {
                $r = new NotasRemision();
            }
            $r->fecha = $request->fecha;
            $r->vendedor_id = $request->vendedor_id;
            $r->estado = 'G';
            $r->cant_total = $request->cant_total;
            //       $r->almacen_id=$request->;
            $r->user_id = Auth::user()->id;
            $r->obs = $request->obs;
            $r->save();
            foreach ($request->libro_id as $k => $v) {
                $detalle = new DetalleRemision();
                $libro = Libro::findOrFail($v);
                $detalle->obs = $request->observacion[$k];
                $detalle->cantidad = $request->cant[$k];
                $detalle->remision_id = $r->id;
                $detalle->libro_id = $v;
                $detalle->codigo = $libro->codigo;
                $detalle->titulo = $libro->titulo;
                $detalle->volumen = $libro->tomo;
                $detalle->save();

                DB::commit();
                Session::flash('success', 'La nota de remision se guardo correctamente');
            }
        } catch (\Exception $e) {
            DB::rollBack();
            return dd($e);
            Session::flash('error', 'Ocurrio un error desconocido al guardar  la nota de remision');
        }
        return redirect('notas-remision?mes=' . $r->fecha->format('m') . '&anio=' . $r->fecha->format('Y') . '&r=' . $r->id . '&imprimir=true');
        //        return redirect('notas-remision/imprimir/'.$r->id);
    }
    public function imprimir(NotasRemision $remision)
    {
        return view('almacen.remision.imprimir', [
            'remision' => $remision
        ]);
    }

    /**
     * Display the specified resource.
     *
     * @param  \App\NotasRemision  $notasRemision
     * @return \Illuminate\Http\Response
     */


    public function procesar(NotasRemision $remision)
    {

        try {
            DB::beginTransaction();
            $datos = [];
            $errors = [];
            foreach ($remision->detalle as $d) {
                $libro = Libro::find($d->libro_id);
                if (!is_null($libro)) {
                    if ($d->cantidad<=$libro->stock ) {
                        $libro->stock = $libro->stock - $d->cantidad;
                        $libro->save();
                    }
                    else{
                        $errors[]=['libro'=>$libro,'overqty'=> $d->cantidad-$libro->stock];
                    }
                }
            }
            if(count($errors)>0){
                DB::rollBack();
                Session::flash('error', 'No se puede procesar la nota porque existen algunos libros que ya no tienen inventario');
                Session::flash('errorLibros', $errors);
                return redirect('/notas-remision?mes=' . $remision->fecha->format('m') . '&anio=' . $remision->fecha->format('Y'));
            }
            foreach ($remision->detalle as $k => $d) {
                for ($i = 0; $i < $d->cantidad; $i++) {
                    $datos[] = [

                        'fecha_remision' => $remision->fecha->format('Y-m-d'),
                        'remision_id' => $remision->id,
                        'cod_libro' => $d->codigo,
                        'titulo_libro' => $d->titulo,
                        'estado' => 'Remitido',
                        'vendedor_id' => $remision->vendedor_id,
                        'tomo_libro' => $d->volumen,
                        'libro_id' => $d->libro_id,
                        'cargo' => 0,
                        'traspaso' => 0,
                        'fechakardex' => $remision->fecha->format('Y-m-d')
                    ];
                }
            }
            //
            Kardex::insert($datos);

            $remision->estado = 'T';
            $remision->save();


            DB::commit();
            Session::flash('success', 'Se ha procesado  la nota de remision correctamente');
        } catch (Exception $e) {
            DB::rollBack();
            Session::flash('error', 'Ocurrio un error inesperado al procesar la nota de remision');
        }
        return redirect('/notas-remision?mes=' . $remision->fecha->format('m') . '&anio=' . $remision->fecha->format('Y'));
    }
    public function validarAnular(NotasRemision $remision)
    {
        try {

            $kardex = Kardex::validarDisponiblesRemision($remision);
            if ($kardex > 0) {
                return response([
                    'status' => true
                ], 200);
            } else {
                return response()->json([
                    'status' => false
                ], 200);
            }
        } catch (\Exception $e) {
            return response()->json([
                'status' => false
            ], 500);
        }
    }
    public function relacion(Request $request)
    {
        $mes = date('m');
        $anio = date('Y');


        if (isset($request->mes)) {
            $mes = $request->mes;
            $anio = $request->anio;
        }

        $datos = DB::table('view_remision_detalle')->whereMonth('fecha', '=', $mes)
            ->whereYear('fecha', '=', $anio)
            ->where('estado', 'T')
            ->orderBy('id', 'ASC')->get();

        if (isset($request->tipo)) {
            $tipo = $request->tipo;

            $nombrearchivo = 'relacion-notas-remision al 31-' . $mes . '-' . $anio;

            if ($tipo == 'excel') {
                return \Excel::download(new RelacionNotasRemisionExport($datos, $mes, $anio), $nombrearchivo . '.xlsx');
            } elseif ($tipo == 'pdf') {
                $pdf = PDF::loadView('almacen.remision.relacion-exportar', [
                    'datos' => $datos,
                    'estilos' => true,
                    'mes' => $mes,
                    'anio' => $anio
                ]);
                $pdf->setPaper('a4', 'portrait');
                //            $pdf->setPaper([0, 0,650, 270], 'portrait');
                //            $pdf->output();
                //            $dom_pdf = $pdf->getDomPDF();
                //            $pdf->set_paper(array(0,0,1000,3000));

                //            return $pdf->download('saldos_socios.pdf');

                return $pdf->stream($nombrearchivo);
            }
        }

        return view('almacen.remision.relacion', [
            'datos' => $datos,
            'mes' => $mes,
            'anio' => $anio
        ]);
    }
    public function anular(NotasRemision $remision)
    {
        try {
            DB::beginTransaction();
            if (Kardex::validarDisponiblesRemision($remision) <= 0) {
                $kardex = Kardex::where('remision_id', $remision->id)
                    ->whereIn('cargo', [0, 1])
                    ->update(['estado' => 'A']);
                $remision->estado = 'A';
                $remision->save();
                Session::flash('success', 'La nota de remision se anulo correctamente.');
            } else {
                Session::flash('warning', 'Existen items en el kardex que  estan asociados a otras notas.');
            }

            foreach ($remision->detalle as $k => $v) {
                $libro = Libro::where('codigo', $v->codigo)->first();
                $libro->stock = $libro->stock + $v->cantidad;
                $libro->save();
            }
            DB::commit();
        } catch (\Exception $e) {
            DB::rollBack();
            Session::flash('error', 'Exisitio un error en el servidor');
        }
        return redirect('/notas-remision?mes=' . $remision->fecha->format('m') . '&anio=' . $remision->fecha->format('Y'));
    }

    public function show(NotasRemision $remision)
    {
        return view('almacen.partials.notaremision', ['remision' => $remision]);
    }

    /**
     * Show the form for editing the specified resource.
     *
     * @param  \App\NotasRemision  $notasRemision
     * @return \Illuminate\Http\Response
     */
    public function edit(NotasRemision $remision)
    {
        $vendedores = Vendedor::orderBy('nombres', 'ASC')->get();
        $libros = Libro::orderBy('codigo', 'ASC')->get();
        return view('almacen.remision.create_edit', [
            'libros' => $libros,
            'vendedores' => $vendedores,
            'remision' => $remision
        ]);
    }

    /**
     * Update the specified resource in storage.
     *
     * @param  \Illuminate\Http\Request  $request
     * @param  \App\NotasRemision  $notasRemision
     * @return \Illuminate\Http\Response
     */
    public function update(Request $request, NotasRemision $notasRemision)
    {
        //
    }

    /**
     * Remove the specified resource from storage.
     *
     * @param  \App\NotasRemision  $notasRemision
     * @return \Illuminate\Http\Response
     */
    public function destroy(NotasRemision $remision)
    {
        try {
            DB::beginTransaction();

            if ($remision->borrarKardex()) {
                $remision->detalle()->delete();
                $remision->delete();
            }

            DB::commit();
            Session::flash('success', 'Se Elimino la nota de remision correctamente');
        } catch (\Exception $e) {
            DB::rollBack();
            Session::flash('error', 'Ocurrio un error al eliminar la nota de remision');
        }
        return redirect('notas-remision?mes=' . $remision->fecha->format('m') . '&anio=' . $remision->fecha->format('Y'));
    }
}
